/* --------------------------------------------------------------
  WidgetConverter.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/*jshint -W079 */

'use strict';

import Setting from '../entities/Setting'; 
import BoxSetting from '../entities/BoxSetting'; 
import TextBox from '../widgets/TextBox'; 
import SelectBox from '../widgets/SelectBox'
import Colorpicker from '../widgets/Colorpicker';
import Switch from '../widgets/Switch'; 
import Title from '../widgets/Title'; 
import Image from '../widgets/Image';

export default {
	/**
	 * Convert a single setting object to its' widget instance.
	 *
	 * @param {Object} setting Comes from the JSON style configuration and contains the setting data.
	 * @param {StyleConfiguration} styleConfiguration The style configuration object that the setting belongs to.
	 * @param {String[]} templateImages Contains a list of the available template images.
	 *
	 * @return {*} Returns the corresponding widget instance.
	 */
	convertSettingToWidget(setting, styleConfiguration, templateImages) {
		StyleEdit.Validator.isObject(setting);
		StyleEdit.Validator.isObject(styleConfiguration);
		StyleEdit.Validator.isObject(templateImages);
		
		let settingInstance;
		let widgetInstance;
		
		settingInstance = (setting.position === undefined) ? new Setting() : new BoxSetting();
		
		settingInstance
			.setName(setting.name)
			.setTitle(StyleEdit.Language.translate(setting.name, 'template'))
			.setType(setting.type);
		
		if (setting.value !== undefined) {
			settingInstance.setValue(setting.value);
		}
		
		if (setting.defaultMetric !== undefined) {
			settingInstance.setDefaultMetric(setting.defaultMetric);
		}
		
		if (setting.position !== undefined) {
			settingInstance.setPosition(setting.position);
		}
		
		if (setting.options !== undefined) {
			settingInstance.setOptions(setting.options);
		}
		
		if (setting.tags !== undefined) {
			settingInstance.setTags(setting.tags);
		}
		
		// Create the widget instance.
		switch (setting.type) {
			case 'color':
				widgetInstance = new Colorpicker(settingInstance, styleConfiguration);
				break;
			case 'text':
				widgetInstance = new TextBox(settingInstance, styleConfiguration);
				break;
			case 'select':
				widgetInstance = new SelectBox(settingInstance, styleConfiguration);
				break;
			case 'switch':
				widgetInstance = new Switch(settingInstance, styleConfiguration);
				break;
			case 'title':
				widgetInstance = new Title(settingInstance);
				break;
			case 'image':
				widgetInstance = new Image(settingInstance, styleConfiguration, templateImages);
				break;
			default:
				throw new Error('Invalid setting setting type detected: ' + setting.type);
		}
		
		return widgetInstance;
	}
	
}
